package topdown.concrete_operator;

import topdown.data_structures.*;
import topdown.term.Term;
import topdown.term.Variable;

import java.util.HashMap;
import java.util.List;

public class ConcreteAntiJoin implements ConcreteOperator {

    private List<ConcreteOperator> operators;
    private List<List<Term>> inputV;
    private List<Term> outputV;
    private Tuple currentTuple;

    public ConcreteAntiJoin(
            List<ConcreteOperator> operators, List<List<Term>> inputV, List<Term> outputV
    ) {
        this.operators = operators;
        this.inputV = inputV;
        this.outputV = outputV;

        // currentTuple represents one tuple from the positive side of an Antijoin
        this.currentTuple = operators.get(0).next();
    }

    @Override
    public Tuple next() {
        getNextMatching();

        if (currentTuple == null) return null;

        // based on inputV and outputV substitute values into the resulting tuple
        Tuple result = new Tuple();
        HashMap<Variable, Term> subst = new HashMap<>();
        for (int j = 0; j < currentTuple.size(); j++) {
            subst.put((Variable) inputV.get(0).get(j), currentTuple.get(j));
        }
        for (Term term : outputV) {
            result.add(term.substitute(subst));
        }

        currentTuple = operators.get(0).next();

        return result;
    }

    @Override
    public void reset() {
        for (ConcreteOperator op : operators) {
            op.reset();
        }
        currentTuple = operators.get(0).next();
    }

    @Override
    public ConcreteOperator instance() {
        ConcreteAntiJoin result = new ConcreteAntiJoin(
                operators, inputV, outputV
        );
        result.reset();

        return result;
    }

    /**
     * Skip tuples from the positive operator, which match input vector
     * constraints (equality of variables specified in inputV for each operator)
     * */
    private void getNextMatching() {
        while (currentTuple != null) {
            boolean shouldSkipTuple = false;

            HashMap<Variable, Term> subst = new HashMap<>();
            for (int j = 0; j < currentTuple.size(); j++) {
                subst.put((Variable) inputV.get(0).get(j), currentTuple.get(j));
            }

            for (int i = 1; i < operators.size(); i++) {
                operators.get(i).reset();
                Tuple tuple2 = operators.get(i).next();

                while (tuple2 != null) {
                    boolean isMatch = true;

                    for (int j = 0; j < inputV.get(i).size(); j++) {
                        Term term = inputV.get(i).get(j);
                        if (subst.containsKey(term) && !subst.get(term).equals(tuple2.get(j))) {
                            isMatch = false;
                            break;
                        }
                    }

                    if (isMatch) {
                        shouldSkipTuple = true;
                        break;
                    }

                    tuple2 = operators.get(i).next();
                }

                if (shouldSkipTuple) break;
            }

            if (!shouldSkipTuple) return;
            currentTuple = operators.get(0).next();
        }
    }
}